package easik.sketch.constraint;


import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.LinkedList;

import javax.swing.tree.DefaultMutableTreeNode;

import org.jgrapht.graph.DefaultEdge;
import org.jgrapht.graph.ListenableDirectedGraph;
import org.jgraph.graph.GraphConstants;

import easik.Easik;
import easik.sketch.edge.GuideEdge;
import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;
import easik.sketch.util.SketchAdapter;
import easik.sketch.vertex.EntityNode;
import easik.sketch.vertex.SketchVertex;
import easik.states.LoadingState;

/** 
 * This is the superclass for all constraints. Constraints get a non-unique label
 * that shows what kind of constraint it is (symbolically) and have a set of paths
 * which it deals with.
 *
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-07-26 Kevin Green
 */
public abstract class Constraint extends SketchVertex {
	/**
	 * The label on the constraint
	 */
	protected String _constraintLabel;
	/**
	 * The paths involved in the constraint
	 */
	protected ArrayList<SketchPath> _paths;
	/**
	 * The edge involved in the constraint
	 */
	protected ArrayList<SketchEdge> _edges;
	/**
	 * The DefaultMutableTreeNode respresentation of this constraint
	 */
	protected DefaultMutableTreeNode _constraintNode;
	/**
	 * Stores whether the constraint is visible or not
	 */
	protected boolean _isVisible = true;
	/**
	 * The Guide Edges of the constraint
	 */
	protected ArrayList<GuideEdge> _visuals;
	
	/**
	 * Default constructor needed for subclasses
	 */
	public Constraint() {
		_visuals = new ArrayList<GuideEdge>();
	}
	
	/**
	 * Creates a very simple constraint node.
	 * 
	 * @param name The label on the node
	 * @param x x coordinate
	 * @param y y coordinate
	 * @param isVisible If the constraint is visible in the graph or not
	 */
	public Constraint(String name, int x, int y, boolean isVisible ){
		super(name, x, y);
		_isVisible = isVisible;
		_visuals = new ArrayList<GuideEdge>();
	}
	
	/**
	 * Returns if the constraint is visible or not (True = visible)
	 * 
	 * @return If the constraint is visible or not
	 */
	public boolean isVisible(){
		return _isVisible;
	}
	
	/**
	 * Sets if the constraint should be visible or not
	 * 
	 * @param inIsVisible If the constraint should be visible or not.
	 */
	public void setVisible(boolean inIsVisible){
		if(inIsVisible != _isVisible){
			_isVisible = inIsVisible;
			Easik.getInstance().getFrame().getSketch().setDirty(true);
		}
		ListenableDirectedGraph<SketchVertex, DefaultEdge> inGraph = Easik.getInstance().getFrame().getSketch().getGraphData();
		SketchAdapter adapter = Easik.getInstance().getFrame().getSketch().getAdapter();
		
		inGraph.removeAllEdges(_visuals);
		inGraph.removeVertex(this);
		
		if(_isVisible){
			addVisualsToSketch(inGraph, adapter);
		}

		Easik.getInstance().getFrame().getSketch().getGraphLayoutCache().reload();
	}
		
	/**
	 * Adds the visual aids to the sketch provided
	 * 
	 * @param inGraph	The graph which will be getting the visual aids
	 * @param adapter	The adapter between the graph and the sketch
	 */
	public void addVisualsToSketch(ListenableDirectedGraph<SketchVertex, DefaultEdge> inGraph,	SketchAdapter adapter)
	{
		//Push loading state
		Easik.getInstance().getStateManager().pushState(new LoadingState());
		
		inGraph.addVertex(this);
		_visuals.clear();
		Rectangle2D bounds = GraphConstants.getBounds(adapter.getVertexCell(this).getAttributes());
		
		// Get Nodes involved in constraint
		ArrayList<SketchVertex> entitiesInvolved = new ArrayList<SketchVertex>();
		for(int i = 0; i < _edges.size(); i++) 
		{		
			SketchVertex source = Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeSource(_edges.get(i));
			SketchVertex target = Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeTarget(_edges.get(i));
			if(!entitiesInvolved.contains(source))
				entitiesInvolved.add(source);
			if(!entitiesInvolved.contains(target))
				entitiesInvolved.add(target);
		}
				
		//Place this object and draw edges going to each entity involved
		int avgX = 0, avgY = 0;
		for(int i=0; i<entitiesInvolved.size(); i++)
		{
			GuideEdge myEdge = new GuideEdge(isDomainOrCoDomain((EntityNode)entitiesInvolved.get(i)));
			_visuals.add(myEdge);
			inGraph.addEdge(this, (SketchVertex)entitiesInvolved.get(i), myEdge);
			avgX += ((SketchVertex)entitiesInvolved.get(i)).getX();
			avgY += ((SketchVertex)entitiesInvolved.get(i)).getY();
		}
		
		//If only one entity is involved add in space
		if(entitiesInvolved.size() == 1)
			avgY += 10;
		
		//If visual does not already have a position then position it
		if(this._posX == 0 && this._posY == 0)
		{
			this.setX(avgX / entitiesInvolved.size());
			this.setY(avgY / entitiesInvolved.size());
		}
		
		bounds.setRect(
			this.getX(),
			this.getY(),
			bounds.getWidth(),
			bounds.getHeight());
		
		//Pop state
		Easik.getInstance().getStateManager().popState();
	}
	
	/**
	 * Tests to see if an entity is either the domain or co-domain of
	 * any of the paths involved in the constraint
	 * 
	 * @param inEntity The entity to be tested
	 * @return True if it is either domain or co-domain of a path involved in the constraint, false otherwise.
	 */
	public boolean isDomainOrCoDomain(EntityNode inEntity){
		int size = _paths.size();
		for(int i=0; i<size; i++){
			SketchPath myPath = (SketchPath)_paths.get(i);
			if(myPath.getDomain() == inEntity || myPath.getCoDomain() == inEntity)
				return true;
		}
		return false;
	}
	
	/**
	 * Adds all edges participating in this constraint to an arraylist of edges.  
	 */
	protected void addEdges(){
		_edges = new ArrayList<SketchEdge>();
		//Fill in _edges array
		for(SketchPath p : _paths){
			LinkedList<SketchEdge> curPathEdges = p.getEdges();
			for(SketchEdge e: curPathEdges)
			{
				if(!_edges.contains(e))
					_edges.add(e);
			}
		}
	}

	/**
	 * Accessor for the constraint label
	 * 
	 * @return The label
	 */
	public String toString() {
		return _constraintLabel;
	}

	/**
	 * Checks to see if the constraint contains a path that uses an edge
	 * 
	 * @param inEdge The edge to check against.
	 * @return True if constraint contains inEdge, false otherwise.
	 */
	public boolean hasEdge(SketchEdge inEdge) {
		return _edges.contains(inEdge);
	}

	/**
	 *	Returns the edges dealt with in the constraint.
	 *
	 * @return The edges
	 */
	public ArrayList<SketchEdge> getEdges() {
		return _edges;
	}

	/**
	 * Returns the set of the paths involved in the constraint
	 * 
	 * @return The array of sketch paths
	 */
	public ArrayList<SketchPath> getPaths() {
		return _paths;
	}
	
	/**
	 * Returns a string corresponding to the constraint type.
	 * (used in XML generation).
	 * 
	 * @return A string of the type of constraint
	 */
	public String getType() {
		if (this instanceof SumConstraint) {
			return "sumconstraint";
		} else if (this instanceof PullbackConstraint) {
			return "pullbackconstraint";
		} else if (this instanceof ProductConstraint) {
			return "productconstraint";
		} else if (this instanceof CommutativeDiagram) {
			return "commutativediagram";
		} else {		
			return "unknown";
		}
	}
	
	/**
	 * Sets the node used to display the constraint in the tree
	 * 
	 * @param inNode The node used to display the constraint in the tree
	 */
	public void setNode(DefaultMutableTreeNode inNode){
		_constraintNode = inNode;
	}
	
	/**
	 * Returns the node used to display the constraint in the tree
	 * 
	 * @return The node used to display the constraint in the tree
	 */
	public DefaultMutableTreeNode getNode(){
		return _constraintNode;
	}
	
	/**
	 * Sets all constraints to visible/invisible in the sketch.
	 * 
	 * @param constraints The list of constraints.
	 * @param show true if constraints are to be set to visible, false otherwise
	 * @since 2006-05-29 Vera Ranieri
	 */
	public static void setAllConstraintsVisible(LinkedList<Constraint> constraints, boolean show){
		for(Constraint c : constraints){
			c.setVisible(show);
		}
	}
	/**
	 * Determines the entities involved in a constraint and lists them for an error message.
	 * @param paths The paths forming the constraint
	 * @return A string of the domains and codomains of the constraint, formatted for an error message
	 * @since 2006-08-04 Vera Ranieri
	 */
	public static String getTablesInvolvedForError(ArrayList<SketchPath> paths){
		ArrayList<String> domains = new ArrayList<String>();
		ArrayList<String> codomains = new ArrayList<String>();
		String domain, codomain;
		for(SketchPath p: paths){
			domain = p.getDomain().getName();
			codomain = p.getCoDomain().getName();
			if(!domains.contains(domain))
				domains.add(domain);
			if(!codomains.contains(codomain))
				codomains.add(codomain);
		}
		String result = "With domains: ";
		for(String d : domains){
			result += d + ", ";
		}
		result = result.substring(0, result.length()-2) + "\n and \n Codomains: ";
		for(String c : codomains){
			result += c +", ";
		}
		result = result.substring(0, result.length()-2);
		return result;
	}
}
